<?php
// webhook.php - Facebook webhook endpoint (verification + message receive)
$cfg = include __DIR__ . '/config.php';

// Ensure data dir exists
if (!is_dir($cfg['DATA_DIR'])) mkdir($cfg['DATA_DIR'], 0755, true);
if (!file_exists($cfg['MESSAGES_FILE'])) file_put_contents($cfg['MESSAGES_FILE'], json_encode([]));
if (!file_exists($cfg['SETTINGS_FILE'])) file_put_contents($cfg['SETTINGS_FILE'], json_encode(['chatgpt_on' => true]));

// Verification (GET)
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Facebook verification: hub.mode, hub.verify_token, hub.challenge
    $mode = $_GET['hub_mode'] ?? $_GET['hub.mode'] ?? null;
    $token = $_GET['hub_verify_token'] ?? $_GET['hub.verify_token'] ?? null;
    $challenge = $_GET['hub_challenge'] ?? $_GET['hub.challenge'] ?? null;

    // Try also common param names
    if (!$mode) $mode = $_GET['hub.mode'] ?? null;
    if (!$token) $token = $_GET['hub.verify_token'] ?? $_GET['verify_token'] ?? null;

    // For simplicity accept any verify token; in production compare to stored token
    if ($mode === 'subscribe') {
        header('Content-Type: text/plain');
        echo $challenge ?: 'OK';
        exit;
    }
    http_response_code(400);
    echo 'Missing verification';
    exit;
}

// Handle POST (message)
$input = file_get_contents('php://input');
$body = json_decode($input, true);
if (!$body) {
    http_response_code(400);
    echo 'Invalid JSON';
    exit;
}

// Process page messages
if (isset($body['object']) && $body['object'] === 'page') {
    foreach ($body['entry'] as $entry) {
        if (!isset($entry['messaging'])) continue;
        foreach ($entry['messaging'] as $event) {
            // Normalize sender id
            $sender = $event['sender']['id'] ?? null;
            if (!$sender) continue;

            // Build message record
            $message_record = [
                'sender' => $sender,
                'timestamp' => time(),
                'raw_event' => $event
            ];

            // Extract text or attachments
            $text = $event['message']['text'] ?? null;
            $attachments = $event['message']['attachments'] ?? null;
            if ($text) $message_record['text'] = $text;
            if ($attachments) $message_record['attachments'] = $attachments;

            // Append to messages.json (group by sender)
            $messages = json_decode(file_get_contents($cfg['MESSAGES_FILE']), true);
            if (!is_array($messages)) $messages = [];
            if (!isset($messages[$sender])) $messages[$sender] = [];
            $messages[$sender][] = $message_record;
            file_put_contents($cfg['MESSAGES_FILE'], json_encode($messages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

            // Check settings: is ChatGPT enabled?
            $settings = json_decode(file_get_contents($cfg['SETTINGS_FILE']), true);
            $chatgpt_on = $settings['chatgpt_on'] ?? true;

            // If ChatGPT is on, process message: if image -> call image match (not implemented here),
            // else do text search via WooCommerce then call OpenAI for response in Bangla.
            if ($chatgpt_on) {
                // Simple flow: if text present -> search woo -> call openai for reply
                if ($text) {
                    require_once __DIR__ . '/woo.php';
                    require_once __DIR__ . '/openai.php';
                    require_once __DIR__ . '/fb_send.php';

                    // Search products by text
                    $products = woo_search_products($text, $cfg);
                    // Build context for OpenAI - instruct to always reply in Bangla
                    $prompt = "You are an e-commerce assistant. Always reply in Bangla.\nCustomer message: {$text}\nProducts: " . json_encode($products);
                    $ai = openai_chat_reply($prompt, $cfg);
                    $reply_text = is_array($ai) && isset($ai['reply_text']) ? $ai['reply_text'] : (is_string($ai) ? $ai : 'দুঃখিত, এখন সাহায্য করতে পারছি না।');

                    // Save bot reply to messages.json
                    $bot_record = [
                        'sender' => 'bot',
                        'to' => $sender,
                        'timestamp' => time(),
                        'text' => $reply_text
                    ];
                    $messages[$sender][] = $bot_record;
                    file_put_contents($cfg['MESSAGES_FILE'], json_encode($messages, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

                    // Send to Facebook
                    fb_send_message($sender, $reply_text, $cfg);
                } else if ($attachments) {
                    // For image attachments, we just acknowledge (image search service can be added)
                    require_once __DIR__ . '/fb_send.php';
                    fb_send_message($sender, 'তুমি যে ছবিটি পাঠিয়েছো সেটা আমি চেক করে দেখছি — বিকল্প পণ্য সাজেস্ট করার ব্যবস্থা আছে।', $cfg);
                }
            } // end chatgpt_on
        } // end each messaging
    } // end each entry

    http_response_code(200);
    echo 'EVENT_RECEIVED';
    exit;
}

http_response_code(404);
echo 'Not a page subscription';
exit;
?>
